<?php

/**
 * vim:set softtabstop=4 shiftwidth=4 expandtab:
 *
 * LICENSE: GNU Affero General Public License, version 3 (AGPL-3.0-or-later)
 * Copyright Ampache.org, 2001-2024
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */

namespace Ampache\Module\Label;

use Ampache\Repository\LabelRepositoryInterface;
use DateTime;

final class LabelListUpdater implements LabelListUpdaterInterface
{
    private LabelRepositoryInterface $labelRepository;

    public function __construct(
        LabelRepositoryInterface $labelRepository
    ) {
        $this->labelRepository = $labelRepository;
    }

    /**
     * Update the labels list based on a comma separated list (ex. label1,label2,label3,..)
     */
    public function update(
        string $labelsComma,
        int $artistId,
        bool $overwrite
    ): bool {
        debug_event(self::class, 'Updating labels for values {' . $labelsComma . '} artist {' . $artistId . '}', 5);

        $clabels      = $this->labelRepository->getByArtist($artistId);
        $filter_list  = preg_split('/(\s*,*\s*)*,+(\s*,*\s*)*/', $labelsComma);
        $editedLabels = (is_array($filter_list)) ? array_unique($filter_list) : [];

        foreach ($clabels as $clid => $clv) {
            if ($clid) {
                $clabel = $this->labelRepository->findById($clid);

                if ($clabel === null) {
                    continue;
                }

                debug_event(self::class, 'Processing label {' . $clabel->name . '}...', 5);
                $found   = false;
                $lstring = '';

                foreach ($editedLabels as $key => $value) {
                    if ($clabel->name == $value) {
                        $found   = true;
                        $lstring = $key;
                        break;
                    }
                }

                if ($found) {
                    debug_event(self::class, 'Already found. Do nothing.', 5);
                    unset($editedLabels[$lstring]);
                } elseif ($overwrite) {
                    debug_event(self::class, 'Not found in the new list. Delete it.', 5);
                    $this->labelRepository->removeArtistAssoc($clabel->getId(), $artistId);
                }
            }
        }

        // Look if we need to add some new labels
        foreach ($editedLabels as $key => $value) {
            if ($value != '') {
                debug_event(self::class, 'Adding new label {' . $value . '}', 4);
                $label_id = $this->labelRepository->lookup($value);
                if ($label_id === 0) {
                    debug_event(self::class, 'Creating a label directly from artist editing is not allowed.', 3);
                }
                if ($label_id > 0) {
                    $clabel = $this->labelRepository->findById($label_id);

                    if ($clabel !== null) {
                        $this->labelRepository->addArtistAssoc($clabel->getId(), $artistId, new DateTime());
                    }
                }
            }
        }

        return true;
    }
}
